import sys
import subprocess
import shutil
from PyQt5.QtWidgets import (
    QApplication, QWidget, QPushButton, QVBoxLayout, QHBoxLayout,
    QTextEdit, QFileDialog, QTabWidget, QListWidget,
    QStyleFactory, QPlainTextEdit, QInputDialog, QMessageBox
)
from PyQt5.QtGui import QPalette, QColor
from PyQt5.QtCore import Qt, QThread, pyqtSignal

class CommandWorker(QThread):
    result_ready = pyqtSignal(str, str, str)

    def __init__(self, cmd):
        super().__init__()
        self.cmd = cmd

    def run(self):
        try:
            result = subprocess.run(self.cmd, stdout=subprocess.PIPE, stderr=subprocess.PIPE, text=True)
            stdout = result.stdout
            stderr = result.stderr
            cmd_str = "$ " + " ".join(self.cmd)
            self.result_ready.emit(stdout, stderr, cmd_str)
        except Exception as e:
            self.result_ready.emit("", f"[!] Error: {str(e)}", "")

class CTFOperatorGUI(QWidget):
    def __init__(self):
        super().__init__()
        self.setWindowTitle("RedOps CTF Injector")
        self.setGeometry(100, 100, 1200, 700)

        QApplication.setStyle(QStyleFactory.create("Fusion"))
        dark_palette = QPalette()
        dark_palette.setColor(QPalette.Window, QColor(30, 30, 30))
        dark_palette.setColor(QPalette.WindowText, Qt.white)
        dark_palette.setColor(QPalette.Base, QColor(45, 45, 45))
        dark_palette.setColor(QPalette.AlternateBase, QColor(60, 60, 60))
        dark_palette.setColor(QPalette.ToolTipBase, Qt.white)
        dark_palette.setColor(QPalette.ToolTipText, Qt.white)
        dark_palette.setColor(QPalette.Text, Qt.white)
        dark_palette.setColor(QPalette.Button, QColor(60, 60, 60))
        dark_palette.setColor(QPalette.ButtonText, Qt.white)
        dark_palette.setColor(QPalette.BrightText, Qt.red)
        dark_palette.setColor(QPalette.Highlight, QColor(0, 122, 204))
        dark_palette.setColor(QPalette.HighlightedText, Qt.black)
        QApplication.setPalette(dark_palette)

        layout = QVBoxLayout()
        self.tabs = QTabWidget()
        self.tabs.setStyleSheet("QTabBar::tab { background: #333; color: white; padding: 10px; } QTabBar::tab:selected { background: #0078d7; }")

        self.tabs.addTab(self.create_editor_tab(), "Editor")
        self.tabs.addTab(self.create_output_tab(), "Output")
        self.tabs.addTab(self.create_script_hub(), "Tool Hub")

        layout.addWidget(self.tabs)
        self.setLayout(layout)

    def create_editor_tab(self):
        editor_tab = QWidget()
        layout = QHBoxLayout()

        left_layout = QVBoxLayout()
        self.output_view = QTextEdit()
        self.output_view.setReadOnly(True)
        self.output_view.setStyleSheet("background-color: #1e1e1e; color: #ffffff; font-family: Consolas;")
        self.output_view.setPlaceholderText("[ Output summary and results will appear here ]")

        btn_layout = QHBoxLayout()
        self.execute_btn = QPushButton("EXECUTE")
        self.clear_btn = QPushButton("CLEAR")
        self.open_btn = QPushButton("OPEN FILE")
        self.full_summary_btn = QPushButton("FULL SUMMARY")
        self.full_summary_btn.clicked.connect(self.show_full_output)

        btn_layout.addWidget(self.execute_btn)
        btn_layout.addWidget(self.clear_btn)
        btn_layout.addWidget(self.open_btn)
        btn_layout.addWidget(self.full_summary_btn)

        left_layout.addWidget(self.output_view)
        left_layout.addLayout(btn_layout)

        self.terminal_input = QPlainTextEdit()
        self.terminal_input.setStyleSheet("background-color: #1e1e1e; color: #00ff00; font-family: Consolas;")
        self.terminal_input.setPlaceholderText("[CTF Terminal Ready]\n$ type your command here and click EXECUTE")

        layout.addLayout(left_layout, 2)
        layout.addWidget(self.terminal_input, 1)

        editor_tab.setLayout(layout)
        return editor_tab

    def show_full_output(self):
        self.output_view.setPlainText(self.last_stdout if hasattr(self, 'last_stdout') else "[No output available]")

    def create_output_tab(self):
        output_tab = QWidget()
        layout = QVBoxLayout()
        self.output = QTextEdit()
        self.output.setReadOnly(True)
        layout.addWidget(self.output)
        output_tab.setLayout(layout)
        return output_tab

    def create_script_hub(self):
        hub_tab = QWidget()
        layout = QHBoxLayout()

        self.tool_list = QListWidget()
        self.tool_list.addItems([
            "volatility3", "strings", "yara", "binwalk", "file", "hexedit", "foremost",
            "exiftool", "bulk_extractor", "tshark", "tcpflow", "chaosreader",
            "nmap", "sqlmap", "hydra", "john", "gobuster", "enum4linux", "hashcat",
            "steghide", "zsteg"
        ])
        self.tool_list.itemClicked.connect(self.run_selected_tool)

        layout.addWidget(self.tool_list)
        hub_tab.setLayout(layout)
        return hub_tab

    def run_selected_tool(self, item):
        tool = item.text()

        if tool == "gobuster":
            if shutil.which("gobuster") is None:
                QMessageBox.critical(self, "Gobuster Missing", "Gobuster is not installed or not in PATH.")
                return
            url, ok1 = QInputDialog.getText(self, "Target URL/IP", "Enter full URL (e.g. https://target.com):")
            wordlist, ok2 = QFileDialog.getOpenFileName(self, "Select Wordlist", "", "Text Files (*.txt)")
            threads, ok3 = QInputDialog.getInt(self, "Threads", "Enter number of threads:", 50, 1, 200)
            extensions, ok4 = QInputDialog.getText(self, "Extensions", "Enter extensions (e.g. php,html):", text="php,html")
            expanded, _ = QInputDialog.getItem(self, "Show Full URLs?", "Enable -e expanded mode?", ["yes", "no"], 0, False)
            skip_tls, _ = QInputDialog.getItem(self, "Skip TLS Verify?", "Enable -k insecure mode?", ["yes", "no"], 0, False)

            if ok1 and ok2 and ok3 and ok4:
                cmd = ["gobuster", "dir", "-u", url.strip(), "-w", wordlist.strip(), "-t", str(threads)]
                if extensions.strip(): cmd += ["-x", extensions.strip()]
                if expanded == "yes": cmd += ["-e"]
                if skip_tls == "yes": cmd += ["-k"]
                self.execute_command(cmd)
            return

        elif tool in ["strings", "yara", "binwalk", "file", "hexedit", "foremost", "exiftool", "bulk_extractor", "tshark", "tcpflow", "chaosreader", "zsteg"]:
            path, ok = QFileDialog.getOpenFileName(self, f"Select file for {tool}", "")
            if ok:
                self.execute_command([tool, path])
            return

        elif tool == "volatility3":
            image_path, _ = QFileDialog.getOpenFileName(self, "Select Memory Image", "", "Raw Files (*.raw *.bin *.mem)")
            plugins = ["windows.pslist", "windows.info", "windows.netscan", "windows.malfind", "windows.cmdline"]
            plugin, ok = QInputDialog.getItem(self, "Select Plugin", "Volatility3 Plugin:", plugins, 0, False)
            if image_path and ok and plugin:
                self.execute_command(["volatility3", "-f", image_path, plugin])
            return

        elif tool == "john":
            hash_file, ok = QFileDialog.getOpenFileName(self, "Select Hash File", "", "Text Files (*.txt *.hash)")
            if ok:
                self.execute_command(["john", hash_file])
            return

        elif tool == "hydra":
            ip, ok1 = QInputDialog.getText(self, "Target IP", "Enter IP:")
            service, ok2 = QInputDialog.getItem(self, "Service", "Select Service:", ["ftp", "ssh", "telnet", "http-get", "http-post"], 0, False)
            userlist, ok3 = QFileDialog.getOpenFileName(self, "Select Username List", "")
            passlist, ok4 = QFileDialog.getOpenFileName(self, "Select Password List", "")
            if ok1 and ok2 and ok3 and ok4:
                self.execute_command(["hydra", "-L", userlist, "-P", passlist, ip, service])
            return

        elif tool == "sqlmap":
            url, ok = QInputDialog.getText(self, "Target URL", "Enter URL:")
            if ok:
                self.execute_command(["sqlmap", "-u", url, "--batch", "--random-agent"])
            return

        elif tool == "nmap":
            target, ok1 = QInputDialog.getText(self, "Target Host/IP", "Enter IP or Host:")
            flags, ok2 = QInputDialog.getText(self, "Nmap Flags", "Enter additional flags:", text="-sV -A")
            if ok1:
                self.execute_command(["nmap"] + flags.split() + [target])
            return

        elif tool == "enum4linux":
            ip, ok = QInputDialog.getText(self, "Target IP", "Enter IP:")
            if ok:
                self.execute_command(["enum4linux", ip])
            return

        elif tool == "hashcat":
            hash_file, ok1 = QFileDialog.getOpenFileName(self, "Select Hash File", "")
            wordlist, ok2 = QFileDialog.getOpenFileName(self, "Select Wordlist", "")
            mode, ok3 = QInputDialog.getText(self, "Attack Mode", "Enter Mode ID:")
            if ok1 and ok2 and ok3:
                self.execute_command(["hashcat", "-m", mode, hash_file, wordlist])
            return

        elif tool == "steghide":
            file, ok1 = QFileDialog.getOpenFileName(self, "Select Stego File", "")
            password, ok2 = QInputDialog.getText(self, "Password", "Enter Password (if any):")
            if ok1:
                cmd = ["steghide", "extract", "-sf", file]
                if ok2 and password:
                    cmd += ["-p", password]
                self.execute_command(cmd)
            return

        self.execute_command([tool])

    def execute_command(self, cmd):
        self.output_view.setPlainText("[ Executing... please wait ]")
        self.worker = CommandWorker(cmd)
        self.worker.result_ready.connect(self.handle_command_result)
        self.worker.start()

    def handle_command_result(self, stdout, stderr, cmd_str):
        self.last_stdout = stdout
        self.terminal_input.appendPlainText(cmd_str)
        self.terminal_input.appendPlainText(stdout)
        self.terminal_input.appendPlainText(stderr)
        summary = self.summarize_output(stdout)
        self.output_view.setPlainText(summary)

    def summarize_output(self, output):
        lines = output.splitlines()
        highlights = [line for line in lines if any(k in line.lower() for k in [
            'error', 'warning', 'alert', 'found', 'malware', 'success', 'hash', 'result'
        ])]
        return "\n".join(highlights[:20] if highlights else lines[:20]) + ("\n... (truncated)" if len(lines) > 20 else "")

if __name__ == '__main__':
    app = QApplication(sys.argv)
    gui = CTFOperatorGUI()
    gui.show()
    sys.exit(app.exec_())
