"use client";

import { MapContainer, TileLayer, Marker, Popup, useMap } from "react-leaflet";
import L from "leaflet";
import "leaflet/dist/leaflet.css";
import "leaflet.heat";
import React, { useEffect, useState } from "react";
import { generateMockPeople } from "@/lib/generateMockPeople";

const userIcon = new L.Icon({
  iconUrl: "https://cdn-icons-png.flaticon.com/512/64/64113.png",
  iconSize: [32, 32],
  iconAnchor: [16, 32],
  popupAnchor: [0, -32],
});

function HeatmapLayer({ points }: { points: Array<[number, number, number]> }) {
  const map = useMap();

  useEffect(() => {
    if (!map || !(L as any).heatLayer) return;

    let heatLayer: any = (map as any)._heatLayer;
    if (heatLayer) {
      map.removeLayer(heatLayer);
    }

    heatLayer = (L as any)
      .heatLayer(points, {
        radius: 35,
        blur: 25,
        maxZoom: 17,
        gradient: {
          0.1: "blue",
          0.3: "lime",
          0.5: "yellow",
          0.8: "red",
        },
      })
      .addTo(map);

    (map as any)._heatLayer = heatLayer;

    return () => {
      if (map && heatLayer) map.removeLayer(heatLayer);
    };
  }, [map, points]);

  return null;
}

export default function CrowdedMap({
  userLocation,
  crowdedAreas,
  onTopCrowdedUpdate,
}: {
  userLocation: { lat: number; lng: number } | null;
  crowdedAreas: {
    name: string;
    crowd_level: string;
    people_count: number;
    lat: number;
    lng: number;
  }[];
  onTopCrowdedUpdate?: (
    top3: { lat: number; lng: number; people_count: number }[]
  ) => void;
}) {
  const [heatPoints, setHeatPoints] = useState<[number, number, number][]>([]);
  const [crowdedAreasState, setCrowdedAreas] = useState<
    { lat: number; lng: number; people_count: number }[]
  >([]);

  useEffect(() => {
    if (!userLocation) return;

    const updateHeat = () => {
      const mock = generateMockPeople(userLocation.lat, userLocation.lng);

      const data: [number, number, number][] = mock.map(
        ({ lat, lng, people_count }) => [
          lat,
          lng,
          Math.min(people_count / 10, 1), //intensity
        ]
      );

      setHeatPoints(data);
      setCrowdedAreas(mock);

      const grouped = mock.reduce((acc, person) => {
        const { name, lat, lng } = person;
        if (!acc[name]) {
          acc[name] = {
            name,
            lat,
            lng,
            people_count: 0,
          };
        }
        acc[name].people_count += 1;
        return acc;
      }, {} as Record<string, { name: string; lat: number; lng: number; people_count: number }>);

      const top3 = Object.values(grouped)
        .sort((a, b) => b.people_count - a.people_count)
        .slice(0, 3);

      if (onTopCrowdedUpdate) {
        onTopCrowdedUpdate(top3);
      }
    };

    updateHeat();
    const interval = setInterval(updateHeat, 5000);
    return () => clearInterval(interval);
  }, [userLocation]);

  return (
    <MapContainer
      center={
        userLocation
          ? [userLocation.lat, userLocation.lng]
          : [13.7456, 100.5347]
      }
      zoom={14}
      style={{ height: "100%", width: "100%" }}
      className="rounded-lg">
      <TileLayer
        url="https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png"
        attribution="&copy; OpenStreetMap contributors"
      />
      {userLocation && (
        <Marker position={[userLocation.lat, userLocation.lng]} icon={userIcon}>
          <Popup>คุณอยู่ที่นี่</Popup>
        </Marker>
      )}
      <HeatmapLayer points={heatPoints} />
    </MapContainer>
  );
}
