"use client"

import { useState } from "react"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Textarea } from "@/components/ui/textarea"
import { Badge } from "@/components/ui/badge"
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs"
import { Alert, AlertDescription } from "@/components/ui/alert"
import {
  Shield,
  MapPin,
  MessageSquare,
  Camera,
  Phone,
  Mail,
  AlertTriangle,
  CheckCircle,
  XCircle,
  Search,
  Upload,
} from "lucide-react"

export default function AISecurityApp() {
  const [scanResult, setScanResult] = useState<any>(null)
  const [isScanning, setIsScanning] = useState(false)
  const [textInput, setTextInput] = useState("")
  const [imageFile, setImageFile] = useState<File | null>(null)

  // Mock scam detection function
  const detectScam = async (text: string) => {
    setIsScanning(true)
    await new Promise((resolve) => setTimeout(resolve, 2000)) // Simulate processing

    const scamKeywords = ["โอนเงิน", "รีบด่วน", "รางวัล", "ฟรี", "คลิกลิงก์", "ธนาคาร", "ตำรวจ", "ลงทุน", "กำไร"]
    const foundKeywords = scamKeywords.filter((keyword) => text.includes(keyword))

    const riskLevel = foundKeywords.length > 2 ? "high" : foundKeywords.length > 0 ? "medium" : "low"

    setScanResult({
      type: "text",
      riskLevel,
      foundKeywords,
      confidence: Math.random() * 100,
      scamType: foundKeywords.length > 0 ? "Financial Scam" : "Safe Content",
    })
    setIsScanning(false)
  }

  // Mock image analysis
  const analyzeImage = async (file: File) => {
    setIsScanning(true)
    await new Promise((resolve) => setTimeout(resolve, 3000))

    const fakeProfiles = ["fake_profile_1.jpg", "scammer_photo.png", "stolen_image.jpg"]
    const isScam = Math.random() > 0.6

    setScanResult({
      type: "image",
      riskLevel: isScam ? "high" : "low",
      isReversed: true,
      originalSource: isScam ? "Found in scam database" : "Original image",
      confidence: Math.random() * 100,
      scamType: isScam ? "Fake Profile" : "Legitimate Image",
    })
    setIsScanning(false)
  }

  const riskAreas = [
    { name: "ตลาดจตุจักร", risk: "high", crimes: 12, type: "Phishing Hotspot" },
    { name: "สถานีรถไฟหัวลำโพง", risk: "medium", crimes: 5, type: "Call Center Scam" },
    { name: "เซ็นทรัลเวิลด์", risk: "low", crimes: 2, type: "Social Media Scam" },
    { name: "สนามบินสุวรรณภูมิ", risk: "medium", crimes: 8, type: "Investment Scam" },
  ]

  return (
    <div className="min-h-screen bg-gradient-to-br from-blue-50 to-indigo-100 p-4">
      <div className="max-w-6xl mx-auto">
        <div className="text-center mb-8">
          <div className="flex items-center justify-center gap-2 mb-4">
            <Shield className="h-8 w-8 text-blue-600" />
            <h1 className="text-3xl font-bold text-gray-900">AI Web Security System</h1>
          </div>
          <p className="text-gray-600">ระบบตรวจจับการหลอกลวงออนไลน์ด้วย AI</p>
        </div>

        <Tabs defaultValue="scan" className="space-y-6">
          <TabsList className="grid w-full grid-cols-4">
            <TabsTrigger value="scan">สแกนหาภัย</TabsTrigger>
            <TabsTrigger value="location">พื้นที่เสี่ยง</TabsTrigger>
            <TabsTrigger value="database">ฐานข้อมูล</TabsTrigger>
            <TabsTrigger value="chatbot">แชทบอท</TabsTrigger>
          </TabsList>

          <TabsContent value="scan" className="space-y-6">
            <div className="grid md:grid-cols-2 gap-6">
              {/* Text Scanner */}
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <MessageSquare className="h-5 w-5" />
                    ตรวจสอบข้อความ
                  </CardTitle>
                </CardHeader>
                <CardContent className="space-y-4">
                  <Textarea
                    placeholder="วางข้อความที่ต้องการตรวจสอบ..."
                    value={textInput}
                    onChange={(e) => setTextInput(e.target.value)}
                    rows={4}
                  />
                  <Button onClick={() => detectScam(textInput)} disabled={!textInput || isScanning} className="w-full">
                    {isScanning ? "กำลังวิเคราะห์..." : "ตรวจสอบข้อความ"}
                  </Button>
                </CardContent>
              </Card>

              {/* Image Scanner */}
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <Camera className="h-5 w-5" />
                    ตรวจสอบรูปภาพ
                  </CardTitle>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div className="border-2 border-dashed border-gray-300 rounded-lg p-6 text-center">
                    <Upload className="h-8 w-8 mx-auto mb-2 text-gray-400" />
                    <Input
                      type="file"
                      accept="image/*"
                      onChange={(e) => {
                        const file = e.target.files?.[0]
                        if (file) {
                          setImageFile(file)
                        }
                      }}
                      className="hidden"
                      id="image-upload"
                    />
                    <label htmlFor="image-upload" className="cursor-pointer">
                      <span className="text-sm text-gray-600">{imageFile ? imageFile.name : "เลือกรูปภาพ"}</span>
                    </label>
                  </div>
                  <Button
                    onClick={() => imageFile && analyzeImage(imageFile)}
                    disabled={!imageFile || isScanning}
                    className="w-full"
                  >
                    {isScanning ? "กำลังวิเคราะห์..." : "ตรวจสอบรูปภาพ"}
                  </Button>
                </CardContent>
              </Card>
            </div>

            {/* Scan Results */}
            {scanResult && (
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <Search className="h-5 w-5" />
                    ผลการตรวจสอบ
                  </CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="space-y-4">
                    <div className="flex items-center gap-2">
                      {scanResult.riskLevel === "high" && <XCircle className="h-5 w-5 text-red-500" />}
                      {scanResult.riskLevel === "medium" && <AlertTriangle className="h-5 w-5 text-yellow-500" />}
                      {scanResult.riskLevel === "low" && <CheckCircle className="h-5 w-5 text-green-500" />}
                      <Badge
                        variant={
                          scanResult.riskLevel === "high"
                            ? "destructive"
                            : scanResult.riskLevel === "medium"
                              ? "secondary"
                              : "default"
                        }
                      >
                        {scanResult.riskLevel === "high"
                          ? "อันตรายสูง"
                          : scanResult.riskLevel === "medium"
                            ? "อันตรายปานกลาง"
                            : "ปลอดภัย"}
                      </Badge>
                    </div>

                    <div className="grid grid-cols-2 gap-4 text-sm">
                      <div>
                        <span className="font-medium">ประเภท:</span> {scanResult.scamType}
                      </div>
                      <div>
                        <span className="font-medium">ความมั่นใจ:</span> {scanResult.confidence.toFixed(1)}%
                      </div>
                    </div>

                    {scanResult.foundKeywords && scanResult.foundKeywords.length > 0 && (
                      <div>
                        <span className="font-medium">คำเสี่ยงที่พบ:</span>
                        <div className="flex flex-wrap gap-1 mt-1">
                          {scanResult.foundKeywords.map((keyword: string, index: number) => (
                            <Badge key={index} variant="outline" className="text-xs">
                              {keyword}
                            </Badge>
                          ))}
                        </div>
                      </div>
                    )}

                    {scanResult.riskLevel === "high" && (
                      <Alert>
                        <AlertTriangle className="h-4 w-4" />
                        <AlertDescription>⚠️ พบสัญญาณการหลอกลวง! แนะนำให้ระวังและไม่ดำเนินการตามข้อความนี้</AlertDescription>
                      </Alert>
                    )}
                  </div>
                </CardContent>
              </Card>
            )}
          </TabsContent>

          <TabsContent value="location">
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <MapPin className="h-5 w-5" />
                  พื้นที่เสี่ยงการหลอกลวง
                </CardTitle>
              </CardHeader>
              <CardContent>
                <div className="space-y-4">
                  {riskAreas.map((area, index) => (
                    <div key={index} className="flex items-center justify-between p-4 border rounded-lg">
                      <div className="flex items-center gap-3">
                        <MapPin className="h-5 w-5 text-gray-500" />
                        <div>
                          <h3 className="font-medium">{area.name}</h3>
                          <p className="text-sm text-gray-600">{area.type}</p>
                        </div>
                      </div>
                      <div className="flex items-center gap-2">
                        <Badge
                          variant={
                            area.risk === "high" ? "destructive" : area.risk === "medium" ? "secondary" : "default"
                          }
                        >
                          {area.risk === "high" ? "เสี่ยงสูง" : area.risk === "medium" ? "เสี่ยงปานกลาง" : "เสี่ยงต่ำ"}
                        </Badge>
                        <span className="text-sm text-gray-500">{area.crimes} คดี</span>
                      </div>
                    </div>
                  ))}
                </div>
              </CardContent>
            </Card>
          </TabsContent>

          <TabsContent value="database">
            <div className="grid md:grid-cols-2 gap-6">
              <Card>
                <CardHeader>
                  <CardTitle>รูปแบบการหลอกลวง (5 ปีย้อนหลัง)</CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="space-y-3">
                    {[
                      { type: "Call Center Scam", icon: Phone, count: 1250, trend: "+15%" },
                      { type: "Phishing Link", icon: Mail, count: 890, trend: "+8%" },
                      { type: "Social Media Scam", icon: MessageSquare, count: 670, trend: "+22%" },
                      { type: "Investment Scam", icon: AlertTriangle, count: 445, trend: "+35%" },
                    ].map((item, index) => (
                      <div key={index} className="flex items-center justify-between p-3 bg-gray-50 rounded">
                        <div className="flex items-center gap-2">
                          <item.icon className="h-4 w-4" />
                          <span className="text-sm">{item.type}</span>
                        </div>
                        <div className="text-right">
                          <div className="text-sm font-medium">{item.count} คดี</div>
                          <div className="text-xs text-red-500">{item.trend}</div>
                        </div>
                      </div>
                    ))}
                  </div>
                </CardContent>
              </Card>

              <Card>
                <CardHeader>
                  <CardTitle>AI Prediction Model</CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="space-y-4">
                    <div className="text-center">
                      <div className="text-2xl font-bold text-blue-600">94.2%</div>
                      <div className="text-sm text-gray-600">ความแม่นยำในการทำนาย</div>
                    </div>
                    <div className="space-y-2">
                      <div className="flex justify-between text-sm">
                        <span>Data Phishing</span>
                        <span>96.1%</span>
                      </div>
                      <div className="flex justify-between text-sm">
                        <span>Financial Scam</span>
                        <span>92.3%</span>
                      </div>
                    </div>
                  </div>
                </CardContent>
              </Card>
            </div>
          </TabsContent>

          <TabsContent value="chatbot">
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <MessageSquare className="h-5 w-5" />
                  AI Security Assistant
                </CardTitle>
              </CardHeader>
              <CardContent>
                <div className="space-y-4">
                  <div className="h-64 border rounded-lg p-4 bg-gray-50 overflow-y-auto">
                    <div className="space-y-3">
                      <div className="bg-blue-100 p-3 rounded-lg max-w-xs">
                        <p className="text-sm">สวัสดีครับ! ผมคือ AI Security Assistant พร้อมช่วยคุณตรวจสอบการหลอกลวงออนไลน์</p>
                      </div>
                      <div className="bg-white p-3 rounded-lg max-w-xs ml-auto">
                        <p className="text-sm">มีคนส่งข้อความมาว่าได้รางวัล 100,000 บาท ต้องโอนค่าธรรมเนียม 500 บาทก่อน</p>
                      </div>
                      <div className="bg-red-100 p-3 rounded-lg max-w-xs">
                        <p className="text-sm">
                          ⚠️ นี่คือการหลอกลวงแบบ Financial Scam! อย่าโอนเงินเด็ดขาด รางวัลจริงไม่ต้องเสียค่าธรรมเนียม
                        </p>
                      </div>
                    </div>
                  </div>
                  <div className="flex gap-2">
                    <Input placeholder="พิมพ์ข้อความเพื่อสอบถาม..." className="flex-1" />
                    <Button>ส่ง</Button>
                  </div>
                </div>
              </CardContent>
            </Card>
          </TabsContent>
        </Tabs>
      </div>
    </div>
  )
}
