"use client"

import type React from "react"

import { useState, useRef } from "react"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Button } from "@/components/ui/button"
import { Alert, AlertDescription } from "@/components/ui/alert"
import { Badge } from "@/components/ui/badge"
import { Smartphone, Camera, Upload, AlertTriangle, Shield, ImageIcon } from "lucide-react"

export default function MobileScanner() {
  const [selectedImage, setSelectedImage] = useState<string | null>(null)
  const [scanResult, setScanResult] = useState<any>(null)
  const [isAnalyzing, setIsAnalyzing] = useState(false)
  const fileInputRef = useRef<HTMLInputElement>(null)

  // จำลองการวิเคราะห์ภาพหน้าจอ
  const analyzeScreenshot = async (imageData: string) => {
    setIsAnalyzing(true)

    // จำลองการประมวลผล OCR
    await new Promise((resolve) => setTimeout(resolve, 2000))

    // จำลองผลการอ่านข้อความจากภาพ
    const mockTexts = [
      "โอนเงิน 1,000 บาท รับรางวัล 50,000 บาท",
      "ธนาคารกรุงเทพ: บัญชีของคุณถูกระงับ กดลิงก์เพื่อยืนยัน",
      "คุณได้รับรางวัลจากการสำรวจ คลิกรับเลย!",
      "ลงทุนบิทคอยน์ กำไร 500% ใน 7 วัน",
      "ตำรวจแจ้งความ: พบหมายจับในนามคุณ",
      "LINE: มีคนส่งข้อความหาคุณ",
      "เนื้อหาปกติ ไม่มีสัญญาณเสี่ยง",
    ]

    const detectedText = mockTexts[Math.floor(Math.random() * mockTexts.length)]

    // วิเคราะห์ความเสี่ยง
    const scamKeywords = ["โอนเงิน", "รางวัล", "ธนาคาร", "ระงับ", "คลิก", "ลิงก์", "ลงทุน", "กำไร", "ตำรวจ", "หมายจับ"]
    const foundKeywords = scamKeywords.filter((keyword) => detectedText.includes(keyword))

    const riskLevel = foundKeywords.length > 2 ? "high" : foundKeywords.length > 0 ? "medium" : "low"

    setScanResult({
      detectedText,
      riskLevel,
      foundKeywords,
      confidence: 85 + Math.random() * 15, // 85-100%
      timestamp: new Date().toLocaleString("th-TH"),
    })

    setIsAnalyzing(false)
  }

  // อัพโหลดภาพหน้าจอ
  const handleImageUpload = (event: React.ChangeEvent<HTMLInputElement>) => {
    const file = event.target.files?.[0]
    if (file) {
      const reader = new FileReader()
      reader.onload = (e) => {
        const imageData = e.target?.result as string
        setSelectedImage(imageData)
        analyzeScreenshot(imageData)
      }
      reader.readAsDataURL(file)
    }
  }

  return (
    <div className="min-h-screen bg-gradient-to-br from-green-50 to-blue-100 p-4">
      <div className="max-w-4xl mx-auto space-y-6">
        {/* Header */}
        <div className="text-center">
          <div className="flex items-center justify-center gap-2 mb-4">
            <Smartphone className="h-8 w-8 text-green-600" />
            <h1 className="text-3xl font-bold text-gray-900">Mobile Screenshot Scanner</h1>
          </div>
          <p className="text-gray-600">อัพโหลดภาพหน้าจอมือถือเพื่อตรวจสอบการหลอกลวง</p>
        </div>

        {/* Upload Section */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Upload className="h-5 w-5" />
              อัพโหลดภาพหน้าจอ
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="space-y-4">
              <div
                className="border-2 border-dashed border-gray-300 rounded-lg p-8 text-center cursor-pointer hover:border-gray-400 transition-colors"
                onClick={() => fileInputRef.current?.click()}
              >
                <ImageIcon className="h-12 w-12 mx-auto mb-4 text-gray-400" />
                <p className="text-gray-600 mb-2">คลิกเพื่อเลือกภาพหน้าจอ</p>
                <p className="text-sm text-gray-500">รองรับไฟล์ JPG, PNG, WebP</p>
              </div>

              <input ref={fileInputRef} type="file" accept="image/*" onChange={handleImageUpload} className="hidden" />

              <div className="text-center">
                <Button onClick={() => fileInputRef.current?.click()} className="flex items-center gap-2">
                  <Camera className="h-4 w-4" />
                  เลือกภาพหน้าจอ
                </Button>
              </div>
            </div>
          </CardContent>
        </Card>

        {/* Image Preview & Results */}
        {selectedImage && (
          <div className="grid md:grid-cols-2 gap-6">
            {/* Image Preview */}
            <Card>
              <CardHeader>
                <CardTitle>ภาพที่อัพโหลด</CardTitle>
              </CardHeader>
              <CardContent>
                <img
                  src={selectedImage || "/placeholder.svg"}
                  alt="Screenshot"
                  className="w-full h-64 object-contain border rounded-lg bg-gray-50"
                />
              </CardContent>
            </Card>

            {/* Analysis Results */}
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <Shield className="h-5 w-5" />
                  ผลการวิเคราะห์
                </CardTitle>
              </CardHeader>
              <CardContent>
                {isAnalyzing ? (
                  <div className="text-center py-8">
                    <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600 mx-auto mb-4"></div>
                    <p className="text-gray-600">กำลังวิเคราะห์ภาพ...</p>
                  </div>
                ) : scanResult ? (
                  <div className="space-y-4">
                    <div className="flex items-center gap-2">
                      <Badge
                        variant={
                          scanResult.riskLevel === "high"
                            ? "destructive"
                            : scanResult.riskLevel === "medium"
                              ? "secondary"
                              : "default"
                        }
                      >
                        {scanResult.riskLevel === "high"
                          ? "อันตรายสูง"
                          : scanResult.riskLevel === "medium"
                            ? "เสี่ยงปานกลาง"
                            : "ปลอดภัย"}
                      </Badge>
                      <span className="text-sm text-gray-500">ความมั่นใจ: {scanResult.confidence.toFixed(1)}%</span>
                    </div>

                    <div>
                      <h4 className="font-medium mb-2">ข้อความที่ตรวจพบ:</h4>
                      <p className="text-sm bg-gray-50 p-3 rounded border">{scanResult.detectedText}</p>
                    </div>

                    {scanResult.foundKeywords.length > 0 && (
                      <div>
                        <h4 className="font-medium mb-2">คำเสี่ยงที่พบ:</h4>
                        <div className="flex flex-wrap gap-1">
                          {scanResult.foundKeywords.map((keyword: string, index: number) => (
                            <Badge key={index} variant="outline" className="text-xs">
                              {keyword}
                            </Badge>
                          ))}
                        </div>
                      </div>
                    )}

                    {scanResult.riskLevel === "high" && (
                      <Alert>
                        <AlertTriangle className="h-4 w-4" />
                        <AlertDescription>
                          ⚠️ <strong>พบสัญญาณการหลอกลวง!</strong>
                          แนะนำให้ระวังและไม่ดำเนินการตามข้อความนี้
                        </AlertDescription>
                      </Alert>
                    )}

                    <div className="text-xs text-gray-500">วิเคราะห์เมื่อ: {scanResult.timestamp}</div>
                  </div>
                ) : null}
              </CardContent>
            </Card>
          </div>
        )}

        {/* Instructions */}
        <Card>
          <CardHeader>
            <CardTitle>วิธีใช้งาน</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="space-y-3 text-sm">
              <div className="flex items-start gap-2">
                <span className="bg-blue-100 text-blue-800 rounded-full w-6 h-6 flex items-center justify-center text-xs font-medium">
                  1
                </span>
                <p>ถ่ายภาพหน้าจอมือถือที่ต้องการตรวจสอบ</p>
              </div>
              <div className="flex items-start gap-2">
                <span className="bg-blue-100 text-blue-800 rounded-full w-6 h-6 flex items-center justify-center text-xs font-medium">
                  2
                </span>
                <p>อัพโหลดภาพผ่านปุ่ม "เลือกภาพหน้าจอ"</p>
              </div>
              <div className="flex items-start gap-2">
                <span className="bg-blue-100 text-blue-800 rounded-full w-6 h-6 flex items-center justify-center text-xs font-medium">
                  3
                </span>
                <p>รอระบบวิเคราะห์และแสดงผลการตรวจสอบ</p>
              </div>
              <div className="flex items-start gap-2">
                <span className="bg-blue-100 text-blue-800 rounded-full w-6 h-6 flex items-center justify-center text-xs font-medium">
                  4
                </span>
                <p>ตรวจสอบระดับความเสี่ยงและคำแนะนำ</p>
              </div>
            </div>
          </CardContent>
        </Card>

        {/* Privacy Notice */}
        <Alert>
          <Shield className="h-4 w-4" />
          <AlertDescription>
            <strong>ความเป็นส่วนตัว:</strong> ภาพที่อัพโหลดจะถูกประมวลผลในเครื่องเท่านั้น ไม่มีการส่งข้อมูลไปยังเซิร์ฟเวอร์ภายนอก
          </AlertDescription>
        </Alert>
      </div>
    </div>
  )
}
